import argparse


parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
parser.add_argument('-l', '--label', default=['pos', 'neg', 'neu'], nargs='+',
                    choices=['pos', 'neg', 'neu'],
                    help='Display events with the specified label(s) (e.g., python main.py -l pos neg)')
parser.add_argument('-t', '--threshold', type=float, default=0,
                    help='A number between [0, 1]. Output events whose maximum confidence score >= the input value.')
parser.add_argument('-f', '--freq', type=int, default=1,
                    help='Display events with frequency >= the input value.')
parser.add_argument('-i', '--infile', default='affevent.csv',
                    help='Input file.')
parser.add_argument('-o', '--outfile', default='affevent.out',
                    help='Output file.')
parser.add_argument('-r', '--readable', action='store_true',
                    help='Display in a more readable format if designated.')
args = parser.parse_args()
print('SETTINGS:')
for arg in vars(args):
    print('{:<20}{}'.format(arg, getattr(args, arg)))
print()


output_list = list()
with open(args.infile, 'r') as fin:
    for fl in fin:
        fs = fl.strip().split(',')
        event = fs[:4]
        label = fs[4]
        score = [float(x) for x in fs[5:8]]
        freq = int(fs[8])
        if any((args.freq and freq < args.freq,
                args.threshold and max(score) < args.threshold + 1e-8,
                args.label and not label in args.label)):
            continue
        output_list.append(fl.strip())

with open(args.outfile, 'w') as fout:
    if not args.readable:
        fout.write('\n'.join(output_list))
    else:
        for fl in output_list:
            fs = fl.split(',')
            event = fs[:4]
            label = fs[4]
            score = [float(x) for x in fs[5:8]]
            freq = int(fs[8])
            tmp = '{:<20}{}\n{:<20}{}\n{:<20}{}\n\n'.format(
                      'Event', ' '.join([x if x else '*' for x in fs[:4]]),
                      'Label', label.upper() + '\t<POS('+'{0:.3g}'.format(score[0])+'), '+\
                                               'NEG('+'{0:.3g}'.format(score[1])+'), '+\
                                               'NEU('+'{0:.3g}'.format(score[2])+')>',
                      'Frequency', str(freq))
            fout.write(tmp)

print('{:d} EVENTS SAVED TO "{}"'.format(len(output_list), args.outfile))



